#ifndef LCD_H
#define LCD_H

/**********************************************************
 Title:   C include file for the LCD library (lcd.c)

 Author:    Mahmood Alimohammadi
 
 Hardware:    AVR ATmega Microcontrollers
 Software: AVR GCC
 Compiler: Atmel Studio 6
 Version 1
 Date: September 24, 2020
 	
DESCRIPTION
This library is written in C and uses 4-bit IO port mode to 
interface with alphanumeric LCDs based on HD44780U. It 
does NOT support 8-bit IO port mode. 
RW pin of the LCD module should be connected to GROUND.
	
Supports 16x1, 16x2, 16x4, and 20x4 Alphanumeric LCDs 
	
***********************************************************
How to Use LCD Interface functions

*	LCD_Clear(): Clear the entire display
*	LCD_Clr_Line(1): Clear the specified line (Here line 1) 
*	LCD_goto(1,1): Go to line 1, position 1
*	LCD_Chr('A'): Display character: A
*	LCD_Text("LCD Interface"): Display text: LCD Interface
*	LCD_Int(number): Display an integer number(positive(e.g. 125) 
	or negative(e.g. -125))

*	LCD_Float(f,2): Display float number f (positive,e.g. 13.14 					
*	or negative,e.g. -13.14), with 2 decimal digits. 
*	2 can be replaced with : 0 to 5 (decimal digits)

*	LCD_Float(f,0): Display float number f (positive,e.g. 1345
*	or negative,e.g. -1345), with 0 (no) decimal digits.
*					
*	LCD_Custom_chr(0, chr0): Generate custom Character 0 
*	at position 0(e.g. man symbol )
*	There are totally 8 character positions-->positions 0 to 7

*	LCD_Chr(0):	Display custom character 0

*	LCD_Scroll_Left(): Scroll the text(on the display) to left
*	LCD_Scroll_Right(): Scroll the text(on the display) to right

 **************************************************

 How to Define the Data Lines and Control Lines
 *
 * There are two port configuration modes: Single_port  and
 * Multiple-port configurations. In the Single-port Configuration, 
 * any port can be used. The four data lines, RS and EN control 
 * lines can be connected to the same port in any order. In the 
 * Single_Port Configuration section below, the lines are all 
 * connected to DDRD. Modify the bit numbers if required. 
 * In the multiple-port configuration, Replace the DDRX of each 
 * bit of the four data lines and RS and En bit number as required 
 * (e.g. lcd_en_ddr	 DDRC). Here, DDRX is replaced with DDRC.
 * Change the bit numbers if needed.
 **************************************************
 
 -----------                   ----------
|ATmega328P |				  |   LCD    |
|           |                 |          |
|           |                 |D0        |
|           |                 |D1        |
|           |                 |D2        |
|           |                 |D3        |
|        PD0|---------------->|D4        |
|        PD1|---------------->|D5        |
|        PD2|---------------->|D6        |
|        PD3|---------------->|D7        |
|           |                 |          |
|        PD4|---------------->|RS        |
|           |          GND--->|RW        |
|        PD5|---------------->|EN        |
 -----------                   ----------
**************************************************
The default configuration is as shown above.

Single-Port Configuration
For Single-port configuration only change the
following define if needed. The bit numbers(0 to 7) 
can also be changed in this mode
**************************************************/
#ifndef DDRX
#define DDRX   DDRD		//(DDRA,DDRB, DDRC, or DDRD)
#endif
/*-----------------------------------------*/
/*
Multiple-Port configuration
If needed, in this mode 
change the DDRX of each bit 
and the bit number(0 to 7)  
*/
#ifndef lcd_d4_ddr
#define lcd_d4_ddr		DDRX	//(DDRA, DDRB, DDRC, or DDRD) 
#endif
#ifndef lcd_d4_bit
#define lcd_d4_bit		0
#endif
#ifndef lcd_d5_ddr
#define lcd_d5_ddr		DDRX    //(DDRA, DDRB, DDRC, or DDRD) 
#endif
#ifndef lcd_d5_bit
#define lcd_d5_bit		1
#endif
#ifndef lcd_d6_ddr
#define lcd_d6_ddr		DDRX	  //(DDRA, DDRB, DDRC, or DDRD) 
#endif
#ifndef lcd_d6_bit
#define lcd_d6_bit		2
#endif
#ifndef lcd_d7_ddr
#define lcd_d7_ddr		DDRX	 //(DDRA, DDRB, DDRC, or DDRD)  
#endif
#ifndef lcd_d7_bit
#define lcd_d7_bit		3
#endif
#ifndef	lcd_rs_ddr
#define lcd_rs_ddr		DDRX   //(DDRA, DDRB, DDRC, or DDRD)  
#endif
#ifndef	lcd_rs_bit
#define lcd_rs_bit		4
#endif
#ifndef	lcd_en_ddr
#define lcd_en_ddr		DDRX    //(DDRA, DDRB, DDRC, or DDRD)
#endif
#ifndef	lcd_en_bit		
#define lcd_en_bit		5 
#endif
/*-----------------------------------------*/
/**
Change the lcd_lines  and lcd_characters 
numbers as needed.
The following define is for 16X2 LCD  
****/
#define lcd_lines		2	//(This can be 1, 2 or 4)

#define lcd_characters		16	//(This can be 16 or 20)
/*-----------------------------------------*/

/**
Display ON and Cursor, OFF, ON and Blink Control
LCD_Cursor 0 = Display ON, Cursor OFF
LCD_Cursor 1 = Display ON, Cursor ON
LCD_Cursor 2 = Display ON, Cursor Character Blink
****/
#define LCD_Cursor 0

/********************************************/
//You should NOT ALTER ANYTHING BEYOND THIS POINT!

//Defines for the LCD command Instructions 
#define lcd_clean		0X01
#define lcd_ddram		0X80
#define lcd_DL8			0X38

/* High nibble of 0X20 for 4-bit mode is 0X02 */
#define lcd_DL4_init	0X02	
#define lcd_DL4			0X28
#define lcd_off			0X08
#define cursor_shift_left	0X06
#define lcd_on_curs_off		0X0C
#define lcd_on_curs_on		0X0E

/*LCD on, Cursor on, Cursor position Character blink*/
#define lcd_on_curs_blink	0X0F	
#define lcd_right_shift		0X1C
#define lcd_left_shift		0X18
#define line_1	0X80	//Line 1 of LCD
#define line_2	0XC0	//Line 2 of LCD
#define line_3	0X90	//Line 3 of 16X4 LCD
#define line_4	0XD0	//Line 4 of 16X4 LCD
#define line3	0X94	//Line 3 of 20X4 LCD
#define line4	0XD4	//Line 4 of 20X4 LCD

/*-----------------------------------------*/
//LCD FUNCTION PROTOTYPES
/**
 Name: LCD_Text
Print a string of characters on the display
Parameter: string-->string of characters to be displayed
Return: None
****/
void LCD_Text(char *string);
/*-----------------------------------------*/
/**
 Name: LCD_Int
Print an integer number on the display
Parameter: number-->the intiger number to be displayed
Return: None
****/
void LCD_Int(int number);
/*-----------------------------------------*/
/*To print float numbers in Atmel Studio 6, or 7, 
go to Project name Properties-->Toolchain-->
AVR/GNU Linker->General and select 
 -Wl,-u,vfprintf
Then click Miscellaneous under the AVR/GNU Linker and 
add the following in the Other Linker Flags box :
-lprintf_flt */


/**
 Name: LCD_Float
Print a floating point number on the display
with or without decimal digits
Parameters: f-->the float number to be displayed
			d-->number of decimal digits. 
			e.g. d=2=two decimal digits, d=0= no decimal digit
Return: None
****/

void LCD_Float(double f,uint8_t d);
/*-----------------------------------------*/
/**
 Name: LCD_Custom_chr
 Generate a custom character 

 Parameters: l-->the location of the character
			 cc--> to be generated
 Return: None
 ****/

void LCD_Custom_chr (uint8_t l, char *cc);
/*-----------------------------------------*/
/**
Name: LCD_Chr
Send character to the display
Parameter: character-->to be sent to the display
Return: None
****/
void LCD_Chr(char character);
/*-----------------------------------------*/
/**
Name: LCD_cmnd
Send  commands to the display
Parameter:cmnd-->to be sent to the display
Return: None
****/
void LCD_cmnd(char cmnd);
/*-----------------------------------------*/
/**
Name: LCD_goto
Set cursor to the specified position
Parameters: y-->line position
		    x--> character position
			x and y start from 1 

Return: None
****/
void LCD_goto(uint8_t y, uint8_t x);
/*-----------------------------------------*/
/**
Name: LCD_Clear
Clear entire display
Cursor at home position
Parameter: None
Return: None
****/
void LCD_Clear(void);
/*-----------------------------------------*/
/**
Name: LCD_Clr_Line
Clear the specified line(1,2,3, or 4) on the display
Parameter: l--> line number
Return: None
****/
void LCD_Clr_Line(uint8_t l);
/*-----------------------------------------*/
/**
Name: LCD_RS_EN_Cntrl_Cmnd
Commands to Clear RS first and then Set and
Clear EN after a delay
Parameter: None
Return:None
****/
void LCD_RS_EN_Cntrl_Cmnd(void);
/*-----------------------------------------*/
/**
Name: LCD_RS_EN_Data_Cmnd
Commands to Set RS  to write data first and 
then Set and Clear EN after a delay
Parameter: None 
Return: None
****/
void LCD_RS_EN_Data_Cmnd(void);

/*-----------------------------------------*/
/**
Name: LCD_Rest_Bits
		Bit 7 low
		Bit 6 low
		Bit 5 low
		Bit 4 low
Parameter: None
Return: None
****/ 
void LCD_Reset_Bits();
/*-----------------------------------------*/
/**
Name: LCD_High_Bits
Send the high bits of the control commands or 
data to the display
Parameter: data--> to be sent to the display
Return: None
****/
void LCD_High_Bits(char data);
/*-----------------------------------------*/
/**
Name: LCD_Low_Bits
Send the low bits of the control command or 
data to the display
Parameter: data to be sent to the display
Return:None
**/
void LCD_Low_Bits(char data);
/*-----------------------------------------*/
/**
Name: LCD_Scroll_Right
Scroll the string of characters to right
Parameter: None
Return: None
****/
void LCD_Scroll_Right(void);
/*-----------------------------------------*/
/**
 Name: LCD_Scroll_Left
Scroll the string of characters to left
Parameter: None
Return: None
****/
void LCD_Scroll_Left(void);
/*-----------------------------------------*/

/**
 Name: LCD_Init
 Clear LCD module, display off
 4bit(DL=0),2line(Bit1=N=1),5x7 pixel(4bit=bit4=0
 LCD on, Cursor off
 Parameter: None
 Return: None
****/
void LCD_init();
/*-----------------------------------------*/
/**
 Name: LCD_on_Curs_on_off_blink
 LCD_Cursor 0 = Display ON, Cursor OFF
 LCD_Cursor 1 = Display ON, Cursor ON
 LCD_Cursor 2 = Display ON, Cursor Character Blink
 Default= Display ON, Cursor OFF	
 
 Parameter: None
 Return: None
****/	

 void LCD_on_Curs_on_off_blink(void);
/*-----------------------------------------*/
#endif //LCD_H
